<?php

/*
  Plugin Name: NitroManager
  version: 1.0.4
  Description: NitroManager, Manage Plugin Updates
 */

// function to activate, deactivate or delete plugin
function nitroPluginAction($args) {
    $returnArr = array();
    if (is_array($args) && count($args)) {
        foreach ($args as $arg) {
            $plugin = isset($arg['plugin']) ? $arg['plugin'] : '';
            if ($plugin) {
                $pluginPath = ABSPATH . 'wp-content/plugins/' . $plugin;
                if (file_exists($pluginPath)) {
                    $action = isset($arg['action']) ? $arg['action'] : '';

                    $isActive = is_plugin_active($plugin);

                    if ($action == 'activate') {
                        if (!$isActive) {
                            try {
                                activate_plugin($pluginPath);
                            } catch (Exception $exc) {
                                $returnArr[$plugin] = $exc->getTraceAsString();
                            }
                            if (is_plugin_active($plugin)) {
                                $returnArr[$plugin] = 'Activated';
                            } else {
                                $returnArr[$plugin] = 'Activation Failed';
                            }
                        } else {
                            $returnArr[$plugin] = 'Active';
                        }
                    } elseif ($action == 'deactivate') {
                        if ($isActive) {
                            try {
                                deactivate_plugins($pluginPath);
                            } catch (Exception $exc) {
                                $returnArr[$plugin] = $exc->getTraceAsString();
                            }
                            if (!is_plugin_active($plugin)) {
                                $returnArr[$plugin] = 'Deactivated';
                            } else {
                                $returnArr[$plugin] = 'Deactivation Failed';
                            }
                        } else {
                            $returnArr[$plugin] = 'Inactive';
                        }
                    } elseif ($action == 'delete') {
                        if (isset($arg['subdirectory'])) {
                            $pluginSubdirectory = ABSPATH . 'wp-content/plugins/' . $arg['subdirectory'];
                            if (file_exists($pluginSubdirectory)) {
                                $deleted = nitro_delete_directory($pluginSubdirectory);
                                if ($deleted === true) {
                                    $returnArr[$plugin] = 'Deleted';
                                } else {
                                    $returnArr[$plugin] = 'Deletion Failed';
                                }
                            } else {
                                $returnArr[$plugin] = 'Not Found';
                            }
                        } else {
                            $returnArr[$plugin] = 'Invalid Subdirectory';
                        }
                    }
                } else {
                    $returnArr[$plugin] = 'Not Found';
                }
            } else {
                return 'Invalid Plugin';
            }
        }
    }
    return $returnArr;
}

// function to get plugin information
function nitroPluginInfo($args) {
    $returnArr = array();
    if (is_array($args) && count($args)) {
        foreach ($args as $arg) {
            $plugin = $arg['plugin'];
            $pluginPath = ABSPATH . 'wp-content/plugins/' . $plugin;
            $returnArr[$plugin] = array();
            $pluginData = get_plugin_data($pluginPath);
            if ($pluginData['Name']) {
                $returnArr[$plugin] = $pluginData;
                $returnArr[$plugin]['is_active'] = 0;
                if (is_plugin_active($plugin)) {
                    $returnArr[$plugin]['is_active'] = 1;
                }
            } else {
                $returnArr[$plugin] = 'Not Found';
            }
        }
    }
    return $returnArr;
}

// function to install or update plugin
function nitroUpdatePlugin($args) {
    $returnArr = array();
    if (is_array($args) && count($args)) {
        foreach ($args as $arg) {
            $plugin = $arg['plugin'];
            $pluginPath = ABSPATH . 'wp-content/plugins/' . $plugin;

            $pluginRoot = ABSPATH . 'wp-content/plugins/';
            $arg['plugin_subdirectory'] = $pluginRoot;

            if (isset($arg['subdirectory']) && $arg['subdirectory']) {
                $arg['plugin_subdirectory'] = $pluginRoot . $arg['subdirectory'] . '/';
            }

            if (is_plugin_active($plugin)) {
                $pluginData = get_plugin_data($pluginPath);
                if ($pluginData['Version'] < $arg['latest_version']) {
                    $installed = nitro_get_plugins($arg);
                    if ($installed) {
                        try {
                            deactivate_plugins($pluginPath);
                        } catch (Exception $exc) {
                            $returnArr[$plugin] = $exc->getTraceAsString();
                        }
                        try {
                            activate_plugin($pluginPath);
                        } catch (Exception $exc) {
                            $returnArr[$plugin] = $exc->getTraceAsString();
                        }
                        if (is_plugin_active($plugin)) {
                            $returnArr[$plugin] = 'Updated';
                        }
                    }
                } else {
                    $returnArr[$plugin] = 'Latest';
                }
            } else {
                $installed = nitro_get_plugins($arg);
                if ($installed) {
                    try {
                        activate_plugin($pluginPath);
                    } catch (Exception $exc) {
                        $returnArr[$plugin] = $exc->getTraceAsString();
                    }
                    if (is_plugin_active($plugin)) {
                        $returnArr[$plugin] = 'Installed';
                    }
                }
            }
        }
    }
    return $returnArr;
}

/* install and update plugin */

// function to process plugin download
function nitro_get_plugins($pluginInfo) {

    $pluginSubdirectory = $pluginInfo['plugin_subdirectory'];
    $destination = $pluginSubdirectory . $pluginInfo['name'] . '.zip';

    if ($pluginInfo['subdirectory']) {
        if (!file_exists($pluginSubdirectory)) {
            @mkdir($pluginSubdirectory, 0777);
        }
    }

    $downloaded = nitro_plugin_download($pluginInfo['update_url'], $destination);

    if ($downloaded) {
        $args = array(
            'path' => $pluginInfo['plugin_subdirectory'],
            'preserve_zip' => $pluginInfo['preserve_zip']
        );
        return nitro_plugin_unpack($args, $args['path'] . $pluginInfo['name'] . '.zip');
    }
}

// function to download plugin
function nitro_plugin_download($url, $path) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $data = curl_exec($ch);
    curl_close($ch);
    if (file_put_contents($path, $data))
        return true;
    else
        return false;
}

// function to unpack downloaded plugin
function nitro_plugin_unpack($args, $target) {
    if ($zip = zip_open($target)) {
        while ($entry = zip_read($zip)) {
            $is_file = substr(zip_entry_name($entry), -1) == '/' ? false : true;
            $file_path = $args['path'] . zip_entry_name($entry);
            if ($is_file) {
                if (zip_entry_open($zip, $entry, "r")) {
                    $fstream = zip_entry_read($entry, zip_entry_filesize($entry));
                    file_put_contents($file_path, $fstream);
                    chmod($file_path, 0777);
                    //echo "save: ".$file_path."<br />";
                }
                zip_entry_close($entry);
            } else {
                if (zip_entry_name($entry)) {
                    @mkdir($file_path);
                    chmod($file_path, 0777);
                    //echo "create: ".$file_path."<br />";
                }
            }
        }
        zip_close($zip);
        if ($args['preserve_zip'] === false) {
            @unlink($target);
        }
        return true;
    }
    return false;
}

// function to delete plugin directory
function nitro_delete_directory($path) {
    return is_file($path) ?
            @unlink($path) :
            array_map(__FUNCTION__, glob($path . '/*')) == @rmdir($path);
}

// check if any username exists in the database
function nitroUsernameExists($args) {
    $username = (string) $args[0];
    if (username_exists($username)) {
        return true;
    }
    return false;
}

// create new user
function nitroCreateUser($args) {
    $user_login = (string) $args[0];
    $user_pass = (string) $args[1];
    $user_email = (string) $args[2];
    $user_role = isset($args[3]) ? (string) $args[3] : 'author';
    if (null == username_exists($user_login)) {
        // Generate the password and create the user
        // $password = wp_generate_password(12, false);
        $user_id = wp_create_user($user_login, $user_pass, $user_email);

        if ($user_id) {
            // Set the nickname
            wp_update_user(
                    array(
                        'ID' => $user_id,
                        'nickname' => $user_login
                    )
            );
            // Set the role
            $user = new WP_User($user_id);
            $user->set_role($user_role);

            return $user;
        }
    } // end if
    return 'User already exists.';
}

/* === code block for adding additional xmlrpc methods === */

// enable xmlrpc if not already
add_filter('xmlrpc_enabled', '__return_true');

// function for adding xmlrpc functions altogether
function add_nitro_xmlrpc_plugin_methods($methods) {
    $methods['nitro.updatePlugin'] = 'nitroUpdatePlugin';
    $methods['nitro.pluginInfo'] = 'nitroPluginInfo';
    $methods['nitro.pluginAction'] = 'nitroPluginAction';
    $methods['nitro.usernameExists'] = 'nitroUsernameExists';
    $methods['nitro.createUser'] = 'nitroCreateUser';
    return $methods;
}

// bind the xmlrpc methods to the main xmlrpc class
add_filter('xmlrpc_methods', 'add_nitro_xmlrpc_plugin_methods');


/**
 * Plugin Update Checker Library 1.2
 * http://w-shadow.com/
 * 
 * Copyright 2012 Janis Elsts
 * Licensed under the GNU GPL license.
 * http://www.gnu.org/licenses/gpl.html
 */
if (!class_exists('PluginUpdateChecker')) {

    /**
     * A custom plugin update checker. 
     * 
     * @author Janis Elsts
     * @copyright 2012
     * @version 1.2
     * @access public
     */
    class PluginUpdateChecker {

        public $metadataUrl = ''; //The URL of the plugin's metadata file.
        public $pluginFile = '';  //Plugin filename relative to the plugins directory.
        public $slug = '';        //Plugin slug.
        public $checkPeriod = 12; //How often to check for updates (in hours).
        public $optionName = '';  //Where to store the update info.
        public $debugMode = false; //Set to TRUE to enable error reporting. Errors are raised using trigger_error()
        //and should be logged to the standard PHP error log.
        private $cronHook = null;
        private $debugBarPlugin = null;

        /**
         * Class constructor.
         * 
         * @param string $metadataUrl The URL of the plugin's metadata file.
         * @param string $pluginFile Fully qualified path to the main plugin file.
         * @param string $slug The plugin's 'slug'. If not specified, the filename part of $pluginFile sans '.php' will be used as the slug.
         * @param integer $checkPeriod How often to check for updates (in hours). Defaults to checking every 12 hours. Set to 0 to disable automatic update checks.
         * @param string $optionName Where to store book-keeping info about update checks. Defaults to 'external_updates-$slug'. 
         */
        public function __construct($metadataUrl, $pluginFile, $slug = '', $checkPeriod = 12, $optionName = '') {
            $this->metadataUrl = $metadataUrl;
            $this->pluginFile = plugin_basename($pluginFile);
            $this->checkPeriod = $checkPeriod;
            $this->slug = $slug;
            $this->optionName = $optionName;
            $this->debugMode = defined('WP_DEBUG') && WP_DEBUG;

            //If no slug is specified, use the name of the main plugin file as the slug.
            //For example, 'my-cool-plugin/cool-plugin.php' becomes 'cool-plugin'.
            if (empty($this->slug)) {
                $this->slug = basename($this->pluginFile, '.php');
            }

            if (empty($this->optionName)) {
                $this->optionName = 'external_updates-' . $this->slug;
            }

            $this->installHooks();
        }

        /**
         * Install the hooks required to run periodic update checks and inject update info 
         * into WP data structures. 
         * 
         * @return void
         */
        protected function installHooks() {
            //Override requests for plugin information
            add_filter('plugins_api', array($this, 'injectInfo'), 20, 3);

            //Insert our update info into the update array maintained by WP
            add_filter('site_transient_update_plugins', array($this, 'injectUpdate')); //WP 3.0+
            add_filter('transient_update_plugins', array($this, 'injectUpdate')); //WP 2.8+

            add_filter('plugin_row_meta', array($this, 'addCheckForUpdatesLink'), 10, 4);
            add_action('admin_init', array($this, 'handleManualCheck'));
            add_action('admin_notices', array($this, 'displayManualCheckResult'));

            //Set up the periodic update checks
            $this->cronHook = 'check_plugin_updates-' . $this->slug;
            if ($this->checkPeriod > 0) {

                //Trigger the check via Cron
                add_filter('cron_schedules', array($this, '_addCustomSchedule'));
                if (!wp_next_scheduled($this->cronHook) && !defined('WP_INSTALLING')) {
                    $scheduleName = 'every' . $this->checkPeriod . 'hours';
                    wp_schedule_event(time(), $scheduleName, $this->cronHook);
                }
                add_action($this->cronHook, array($this, 'checkForUpdates'));

                register_deactivation_hook($this->pluginFile, array($this, '_removeUpdaterCron'));

                //In case Cron is disabled or unreliable, we also manually trigger 
                //the periodic checks while the user is browsing the Dashboard. 
                add_action('admin_init', array($this, 'maybeCheckForUpdates'));
            } else {
                //Periodic checks are disabled.
                wp_clear_scheduled_hook($this->cronHook);
            }

            add_action('plugins_loaded', array($this, 'initDebugBarPanel'));
        }

        /**
         * Add our custom schedule to the array of Cron schedules used by WP.
         * 
         * @param array $schedules
         * @return array
         */
        public function _addCustomSchedule($schedules) {
            if ($this->checkPeriod && ($this->checkPeriod > 0)) {
                $scheduleName = 'every' . $this->checkPeriod . 'hours';
                $schedules[$scheduleName] = array(
                    'interval' => $this->checkPeriod * 3600,
                    'display' => sprintf('Every %d hours', $this->checkPeriod),
                );
            }
            return $schedules;
        }

        /**
         * Remove the scheduled cron event that the library uses to check for updates.
         *
         * @return void
         */
        public function _removeUpdaterCron() {
            wp_clear_scheduled_hook($this->cronHook);
        }

        /**
         * Get the name of the update checker's WP-cron hook. Mostly useful for debugging.
         *
         * @return string
         */
        public function getCronHookName() {
            return $this->cronHook;
        }

        /**
         * Retrieve plugin info from the configured API endpoint.
         * 
         * @uses wp_remote_get()
         * 
         * @param array $queryArgs Additional query arguments to append to the request. Optional.
         * @return PluginInfo
         */
        public function requestInfo($queryArgs = array()) {
            //Query args to append to the URL. Plugins can add their own by using a filter callback (see addQueryArgFilter()).
            $installedVersion = $this->getInstalledVersion();
            $queryArgs['installed_version'] = ($installedVersion !== null) ? $installedVersion : '';
            $queryArgs = apply_filters('puc_request_info_query_args-' . $this->slug, $queryArgs);

            //Various options for the wp_remote_get() call. Plugins can filter these, too.
            $options = array(
                'timeout' => 10, //seconds
                'headers' => array(
                    'Accept' => 'application/json'
                ),
            );
            $options = apply_filters('puc_request_info_options-' . $this->slug, $options);

            //The plugin info should be at 'http://your-api.com/url/here/$slug/info.json'
            $url = $this->metadataUrl;
            if (!empty($queryArgs)) {
                $url = add_query_arg($queryArgs, $url);
            }

            $result = wp_remote_get(
                    $url, $options
            );

            //Try to parse the response
            $pluginInfo = null;
            if (!is_wp_error($result) && isset($result['response']['code']) && ($result['response']['code'] == 200) && !empty($result['body'])) {
                $pluginInfo = PluginInfo::fromJson($result['body'], $this->debugMode);
            } else if ($this->debugMode) {
                $message = sprintf("The URL %s does not point to a valid plugin metadata file. ", $url);
                if (is_wp_error($result)) {
                    $message .= "WP HTTP error: " . $result->get_error_message();
                } else if (isset($result['response']['code'])) {
                    $message .= "HTTP response code is " . $result['response']['code'] . " (expected: 200)";
                } else {
                    $message .= "wp_remote_get() returned an unexpected result.";
                }
                trigger_error($message, E_USER_WARNING);
            }

            $pluginInfo = apply_filters('puc_request_info_result-' . $this->slug, $pluginInfo, $result);
            return $pluginInfo;
        }

        /**
         * Retrieve the latest update (if any) from the configured API endpoint.
         * 
         * @uses PluginUpdateChecker::requestInfo()
         * 
         * @return PluginUpdate An instance of PluginUpdate, or NULL when no updates are available.
         */
        public function requestUpdate() {
            //For the sake of simplicity, this function just calls requestInfo() 
            //and transforms the result accordingly.
            $pluginInfo = $this->requestInfo(array('checking_for_updates' => '1'));
            if ($pluginInfo == null) {
                return null;
            }
            return PluginUpdate::fromPluginInfo($pluginInfo);
        }

        /**
         * Get the currently installed version of the plugin.
         * 
         * @return string Version number.
         */
        public function getInstalledVersion() {
            if (!function_exists('get_plugins')) {
                require_once( ABSPATH . '/wp-admin/includes/plugin.php' );
            }
            $allPlugins = get_plugins();
            if (array_key_exists($this->pluginFile, $allPlugins) && array_key_exists('Version', $allPlugins[$this->pluginFile])) {
                return $allPlugins[$this->pluginFile]['Version'];
            } else {
                //This can happen if the filename is wrong or the plugin is installed in mu-plugins.
                if ($this->debugMode) {
                    trigger_error(
                            sprintf(
                                    "Can't to read the Version header for %s. The filename may be incorrect, or the file is not present in /wp-content/plugins.", $this->pluginFile
                            ), E_USER_WARNING
                    );
                }
                return null;
            }
        }

        /**
         * Check for plugin updates. 
         * The results are stored in the DB option specified in $optionName.
         * 
         * @return PluginUpdate|null
         */
        public function checkForUpdates() {
            $installedVersion = $this->getInstalledVersion();
            //Fail silently if we can't find the plugin or read its header.
            if ($installedVersion === null) {
                if ($this->debugMode) {
                    trigger_error(
                            sprintf('Skipping update check for %s - installed version unknown.', $this->pluginFile), E_USER_WARNING
                    );
                }
                return null;
            }

            $state = $this->getUpdateState();
            if (empty($state)) {
                $state = new StdClass;
                $state->lastCheck = 0;
                $state->checkedVersion = '';
                $state->update = null;
            }

            $state->lastCheck = time();
            $state->checkedVersion = $installedVersion;
            $this->setUpdateState($state); //Save before checking in case something goes wrong 

            $state->update = $this->requestUpdate();
            $this->setUpdateState($state);

            return $this->getUpdate();
        }

        /**
         * Check for updates only if the configured check interval has already elapsed.
         * 
         * @return void
         */
        public function maybeCheckForUpdates() {
            if (empty($this->checkPeriod)) {
                return;
            }
            $state = $this->getUpdateState();

            $shouldCheck = empty($state) ||
                    !isset($state->lastCheck) ||
                    ( (time() - $state->lastCheck) >= $this->checkPeriod * 3600 );

            if ($shouldCheck) {
                $this->checkForUpdates();
            }
        }

        /**
         * Load the update checker state from the DB.
         *  
         * @return StdClass|null
         */
        public function getUpdateState() {
            $state = get_site_option($this->optionName);
            if (!empty($state) && isset($state->update) && !($state->update instanceof PluginUpdate)) {
                $state->update = PluginUpdate::fromObject($state->update);
            }
            return $state;
        }

        /**
         * Persist the update checker state to the DB.
         * 
         * @param StdClass $state
         * @return void
         */
        private function setUpdateState($state) {
            if (isset($state->update) && ($state->update instanceof PluginUpdate)) {
                $update = $state->update;/** @var PluginUpdate $update */
                $state->update = $update->toStdClass();
            }
            update_site_option($this->optionName, $state);
        }

        /**
         * Reset update checker state - i.e. last check time, cached update data and so on.
         *
         * Call this when your plugin is being uninstalled, or if you want to
         * clear the update cache.
         */
        public function resetUpdateState() {
            delete_site_option($this->optionName);
        }

        /**
         * Intercept plugins_api() calls that request information about our plugin and 
         * use the configured API endpoint to satisfy them. 
         * 
         * @see plugins_api()
         * 
         * @param mixed $result
         * @param string $action
         * @param array|object $args
         * @return mixed
         */
        public function injectInfo($result, $action = null, $args = null) {
            $relevant = ($action == 'plugin_information') && isset($args->slug) && ($args->slug == $this->slug);
            if (!$relevant) {
                return $result;
            }

            $pluginInfo = $this->requestInfo();
            $pluginInfo = apply_filters('puc_pre_inject_info-' . $this->slug, $pluginInfo);
            if ($pluginInfo) {
                return $pluginInfo->toWpFormat();
            }

            return $result;
        }

        /**
         * Insert the latest update (if any) into the update list maintained by WP.
         * 
         * @param StdClass $updates Update list.
         * @return StdClass Modified update list.
         */
        public function injectUpdate($updates) {
            //Is there an update to insert?
            $update = $this->getUpdate();
            if (!empty($update)) {
                //Let plugins filter the update info before it's passed on to WordPress.
                $update = apply_filters('puc_pre_inject_update-' . $this->slug, $update);
                $updates->response[$this->pluginFile] = $update->toWpFormat();
            } else {
                if (is_object($updates)) {
                    unset($updates->response[$this->pluginFile]);
                }
            }

            return $updates;
        }

        /**
         * Get the details of the currently available update, if any.
         *
         * If no updates are available, or if the last known update version is below or equal
         * to the currently installed version, this method will return NULL.
         *
         * Uses cached update data. To retrieve update information straight from
         * the metadata URL, call requestUpdate() instead.
         *
         * @return PluginUpdate|null
         */
        public function getUpdate() {
            $state = $this->getUpdateState();/** @var StdClass $state */
            //Is there an update available insert?
            if (!empty($state) && isset($state->update) && !empty($state->update)) {
                $update = $state->update;
                //Check if the update is actually newer than the currently installed version.
                $installedVersion = $this->getInstalledVersion();
                if (($installedVersion !== null) && version_compare($update->version, $installedVersion, '>')) {
                    return $update;
                }
            }
            return null;
        }

        /**
         * Add a "Check for updates" link to the plugin row in the "Plugins" page. By default,
         * the new link will appear after the "Visit plugin site" link.
         *
         * You can change the link text by using the "puc_manual_check_link-$slug" filter.
         * Returning an empty string from the filter will disable the link.
         *
         * @param array $pluginMeta Array of meta links.
         * @param string $pluginFile
         * @param array|null $pluginData Currently ignored.
         * @param string|null $status Currently ignored/
         * @return array
         */
        public function addCheckForUpdatesLink($pluginMeta, $pluginFile, $pluginData = null, $status = null) {
            if ($pluginFile == $this->pluginFile && current_user_can('update_plugins')) {
                $linkText = apply_filters('puc_manual_check_link-' . $this->slug, 'Check for updates');
                $linkUrl = wp_nonce_url(
                        add_query_arg('puc_check_for_updates', $this->slug, admin_url('plugins.php')), 'puc_check_for_updates'
                );
                if (!empty($linkText)) {
                    $pluginMeta[] = sprintf('<a href="%s">%s</a>', esc_attr($linkUrl), $linkText);
                }
            }
            return $pluginMeta;
        }

        /**
         * Check for updates when the user clicks the "Check for updates" link.
         * @see self::addCheckForUpdatesLink()
         *
         * @return void
         */
        public function handleManualCheck() {
            $shouldCheck = isset($_GET['puc_check_for_updates']) && $_GET['puc_check_for_updates'] == $this->slug && current_user_can('update_plugins') && check_admin_referer('puc_check_for_updates');

            if ($shouldCheck) {
                $update = $this->checkForUpdates();
                $status = ($update === null) ? 'no_update' : 'update_available';
                wp_redirect(add_query_arg('puc_update_check_result', $status, admin_url('plugins.php')));
            }
        }

        /**
         * Display the results of a manual update check.
         * @see self::handleManualCheck()
         *
         * You can change the result message by using the "puc_manual_check_message-$slug" filter.
         */
        public function displayManualCheckResult() {
            if (isset($_GET['puc_update_check_result'])) {
                $status = strval($_GET['puc_update_check_result']);
                if ($status == 'no_update') {
                    $message = 'This plugin is up to date.';
                } else if ($status == 'update_available') {
                    $message = 'A new version of this plugin is available.';
                } else {
                    $message = sprintf('Unknown update checker status "%s"', htmlentities($status));
                }
                printf(
                        '<div class="updated"><p>%s</p></div>', apply_filters('puc_manual_check_message-' . $this->slug, $message, $status)
                );
            }
        }

        /**
         * Register a callback for filtering query arguments. 
         * 
         * The callback function should take one argument - an associative array of query arguments.
         * It should return a modified array of query arguments.
         * 
         * @uses add_filter() This method is a convenience wrapper for add_filter().
         * 
         * @param callable $callback
         * @return void
         */
        public function addQueryArgFilter($callback) {
            add_filter('puc_request_info_query_args-' . $this->slug, $callback);
        }

        /**
         * Register a callback for filtering arguments passed to wp_remote_get().
         * 
         * The callback function should take one argument - an associative array of arguments -
         * and return a modified array or arguments. See the WP documentation on wp_remote_get()
         * for details on what arguments are available and how they work. 
         * 
         * @uses add_filter() This method is a convenience wrapper for add_filter().
         * 
         * @param callable $callback
         * @return void
         */
        public function addHttpRequestArgFilter($callback) {
            add_filter('puc_request_info_options-' . $this->slug, $callback);
        }

        /**
         * Register a callback for filtering the plugin info retrieved from the external API.
         * 
         * The callback function should take two arguments. If the plugin info was retrieved 
         * successfully, the first argument passed will be an instance of  PluginInfo. Otherwise, 
         * it will be NULL. The second argument will be the corresponding return value of 
         * wp_remote_get (see WP docs for details).
         *  
         * The callback function should return a new or modified instance of PluginInfo or NULL.
         * 
         * @uses add_filter() This method is a convenience wrapper for add_filter().
         * 
         * @param callable $callback
         * @return void
         */
        public function addResultFilter($callback) {
            add_filter('puc_request_info_result-' . $this->slug, $callback, 10, 2);
        }

        /**
         * Register a callback for one of the update checker filters.
         *
         * Identical to add_filter(), except it automatically adds the "puc_" prefix
         * and the "-$plugin_slug" suffix to the filter name. For example, "request_info_result"
         * becomes "puc_request_info_result-your_plugin_slug".
         *
         * @param string $tag
         * @param callable $callback
         * @param int $priority
         * @param int $acceptedArgs
         */
        public function addFilter($tag, $callback, $priority = 10, $acceptedArgs = 1) {
            add_filter('puc_' . $tag . '-' . $this->slug, $callback, $priority, $acceptedArgs);
        }

        /**
         * Initialize the update checker Debug Bar plugin/add-on thingy.
         */
        public function initDebugBarPanel() {
            if (class_exists('Debug_Bar')) {
                //require_once dirname(__FILE__) . '/debug-bar-plugin.php';
                $this->debugBarPlugin = new PucDebugBarPlugin($this);
            }
        }

    }

}

if (!class_exists('PluginInfo')) {

    /**
     * A container class for holding and transforming various plugin metadata.
     * 
     * @author Janis Elsts
     * @copyright 2012
     * @version 1.0
     * @access public
     */
    class PluginInfo {

        //Most fields map directly to the contents of the plugin's info.json file.
        //See the relevant docs for a description of their meaning.  
        public $name;
        public $slug;
        public $version;
        public $homepage;
        public $sections;
        public $download_url;
        public $author;
        public $author_homepage;
        public $requires;
        public $tested;
        public $upgrade_notice;
        public $rating;
        public $num_ratings;
        public $downloaded;
        public $last_updated;
        public $id = 0; //The native WP.org API returns numeric plugin IDs, but they're not used for anything.

        /**
         * Create a new instance of PluginInfo from JSON-encoded plugin info 
         * returned by an external update API.
         * 
         * @param string $json Valid JSON string representing plugin info.
         * @param bool $triggerErrors
         * @return PluginInfo|null New instance of PluginInfo, or NULL on error.
         */

        public static function fromJson($json, $triggerErrors = false) {
            /** @var StdClass $apiResponse */
            $apiResponse = json_decode($json);
            if (empty($apiResponse) || !is_object($apiResponse)) {
                if ($triggerErrors) {
                    trigger_error(
                            "Failed to parse plugin metadata. Try validating your .json file with http://jsonlint.com/", E_USER_NOTICE
                    );
                }
                return null;
            }

            //Very, very basic validation.
            $valid = isset($apiResponse->name) && !empty($apiResponse->name) && isset($apiResponse->version) && !empty($apiResponse->version);
            if (!$valid) {
                if ($triggerErrors) {
                    trigger_error(
                            "The plugin metadata file does not contain the required 'name' and/or 'version' keys.", E_USER_NOTICE
                    );
                }
                return null;
            }

            $info = new PluginInfo();
            foreach (get_object_vars($apiResponse) as $key => $value) {
                $info->$key = $value;
            }

            return $info;
        }

        /**
         * Transform plugin info into the format used by the native WordPress.org API
         * 
         * @return object
         */
        public function toWpFormat() {
            $info = new StdClass;

            //The custom update API is built so that many fields have the same name and format
            //as those returned by the native WordPress.org API. These can be assigned directly. 
            $sameFormat = array(
                'name', 'slug', 'version', 'requires', 'tested', 'rating', 'upgrade_notice',
                'num_ratings', 'downloaded', 'homepage', 'last_updated',
            );
            foreach ($sameFormat as $field) {
                if (isset($this->$field)) {
                    $info->$field = $this->$field;
                } else {
                    $info->$field = null;
                }
            }

            //Other fields need to be renamed and/or transformed.
            $info->download_link = $this->download_url;

            if (!empty($this->author_homepage)) {
                $info->author = sprintf('<a href="%s">%s</a>', $this->author_homepage, $this->author);
            } else {
                $info->author = $this->author;
            }

            if (is_object($this->sections)) {
                $info->sections = get_object_vars($this->sections);
            } elseif (is_array($this->sections)) {
                $info->sections = $this->sections;
            } else {
                $info->sections = array('description' => '');
            }

            return $info;
        }

    }

}

if (!class_exists('PluginUpdate')) {

    /**
     * A simple container class for holding information about an available update.
     * 
     * @author Janis Elsts
     * @copyright 2012
     * @version 1.2
     * @access public
     */
    class PluginUpdate {

        public $id = 0;
        public $slug;
        public $version;
        public $homepage;
        public $download_url;
        public $upgrade_notice;
        private static $fields = array('id', 'slug', 'version', 'homepage', 'download_url', 'upgrade_notice');

        /**
         * Create a new instance of PluginUpdate from its JSON-encoded representation.
         * 
         * @param string $json
         * @param bool $triggerErrors
         * @return PluginUpdate|null
         */
        public static function fromJson($json, $triggerErrors = false) {
            //Since update-related information is simply a subset of the full plugin info,
            //we can parse the update JSON as if it was a plugin info string, then copy over
            //the parts that we care about.
            $pluginInfo = PluginInfo::fromJson($json, $triggerErrors);
            if ($pluginInfo != null) {
                return PluginUpdate::fromPluginInfo($pluginInfo);
            } else {
                return null;
            }
        }

        /**
         * Create a new instance of PluginUpdate based on an instance of PluginInfo.
         * Basically, this just copies a subset of fields from one object to another.
         * 
         * @param PluginInfo $info
         * @return PluginUpdate
         */
        public static function fromPluginInfo($info) {
            return self::fromObject($info);
        }

        /**
         * Create a new instance of PluginUpdate by copying the necessary fields from 
         * another object.
         *  
         * @param StdClass|PluginInfo|PluginUpdate $object The source object.
         * @return PluginUpdate The new copy.
         */
        public static function fromObject($object) {
            $update = new PluginUpdate();
            foreach (self::$fields as $field) {
                $update->$field = $object->$field;
            }
            return $update;
        }

        /**
         * Create an instance of StdClass that can later be converted back to 
         * a PluginUpdate. Useful for serialization and caching, as it avoids
         * the "incomplete object" problem if the cached value is loaded before
         * this class.
         * 
         * @return StdClass
         */
        public function toStdClass() {
            $object = new StdClass();
            foreach (self::$fields as $field) {
                $object->$field = $this->$field;
            }
            return $object;
        }

        /**
         * Transform the update into the format used by WordPress native plugin API.
         * 
         * @return object
         */
        public function toWpFormat() {
            $update = new StdClass;

            $update->id = $this->id;
            $update->slug = $this->slug;
            $update->new_version = $this->version;
            $update->url = $this->homepage;
            $update->package = $this->download_url;
            if (!empty($this->upgrade_notice)) {
                $update->upgrade_notice = $this->upgrade_notice;
            }

            return $update;
        }

    }

}

if (!class_exists('PucDebugBarPlugin')) {

    class PucDebugBarPlugin {

        /** @var PluginUpdateChecker */
        private $updateChecker;

        public function __construct($updateChecker) {
            $this->updateChecker = $updateChecker;

            add_filter('debug_bar_panels', array($this, 'addDebugBarPanel'));
            add_action('debug_bar_enqueue_scripts', array($this, 'enqueuePanelDependencies'));

            add_action('wp_ajax_puc_debug_check_now', array($this, 'ajaxCheckNow'));
            add_action('wp_ajax_puc_debug_request_info', array($this, 'ajaxRequestInfo'));
        }

        /**
         * Register the PUC Debug Bar panel.
         *
         * @param array $panels
         * @return array
         */
        public function addDebugBarPanel($panels) {
            //require_once dirname(__FILE__) . '/debug-bar-panel.php';
            if (current_user_can('update_plugins') && class_exists('PluginUpdateCheckerPanel')) {
                $panels[] = new PluginUpdateCheckerPanel($this->updateChecker);
            }
            return $panels;
        }

        /**
         * Enqueue our Debug Bar scripts and styles.
         */
        public function enqueuePanelDependencies() {
            add_action('wp_head', 'print_inline_css_script');
            /**
              wp_enqueue_style(
              'puc-debug-bar-style', plugins_url("/css/puc-debug-bar.css", __FILE__), array('debug-bar'), '20121026-3'
              );

              wp_enqueue_script(
              'puc-debug-bar-js', plugins_url("/js/debug-bar.js", __FILE__), array('jquery'), '20121026'
              );
              /* */
        }

        public function print_inline_css_script() {
            ?>
            <style type="text/css">
                .puc-debug-bar-panel pre {
                    margin-top: 0;
                }

                table.puc-debug-data th {
                    width: 11em;
                }

                .puc-ajax-response {
                    border: 1px solid #dfdfdf;
                    border-radius: 3px;
                    padding: 0.5em;
                    margin: 5px 0;
                    background-color: white;
                }

                .puc-ajax-nonce {
                    display: none;
                }
            </style>
            <script type="text/javascript">
                jQuery(function($) {

                    function runAjaxAction(button, action) {
                        button = $(button);
                        var panel = button.closest('.puc-debug-bar-panel');
                        var responseBox = button.closest('td').find('.puc-ajax-response');

                        responseBox.text('Processing...').show();
                        $.post(
                                ajaxurl,
                                {
                                    action: action,
                                    slug: panel.data('slug'),
                                    _wpnonce: panel.data('nonce')
                                },
                        function(data) {
                            responseBox.html(data);
                        },
                                'html'
                                );
                    }

                    $('.puc-debug-bar-panel input[name="puc-check-now-button"]').click(function() {
                        runAjaxAction(this, 'puc_debug_check_now');
                        return false;
                    });

                    $('.puc-debug-bar-panel input[name="puc-request-info-button"]').click(function() {
                        runAjaxAction(this, 'puc_debug_request_info');
                        return false;
                    });


                    // Debug Bar uses the panel class name as part of its link and container IDs. This means we can
                    // end up with multiple identical IDs if more than one plugin uses the update checker library.
                    // Fix it by replacing the class name with the plugin slug.
                    var panels = $('#debug-menu-targets').find('.puc-debug-bar-panel');
                    panels.each(function(index) {
                        var panel = $(this);
                        var slug = panel.data('slug');
                        var target = panel.closest('.debug-menu-target');

                        //Change the panel wrapper ID.
                        target.attr('id', 'debug-menu-target-puc-' + slug);

                        //Change the menu link ID as well and point it at the new target ID.
                        $('#puc-debug-menu-link-' + panel.data('slug'))
                                .closest('.debug-menu-link')
                                .attr('id', 'debug-menu-link-puc-' + slug)
                                .attr('href', '#' + target.attr('id'));
                    });
                });
            </script>
            <?php

        }

        /**
         * Run an update check and output the result. Useful for making sure that
         * the update checking process works as expected.
         */
        public function ajaxCheckNow() {
            if ($_POST['slug'] !== $this->updateChecker->slug) {
                return;
            }
            $this->preAjaxReqest();
            $update = $this->updateChecker->checkForUpdates();
            if ($update !== null) {
                echo "An update is available:";
                echo '<pre>', htmlentities(print_r($update, true)), '</pre>';
            } else {
                echo 'No updates found.';
            }
            exit;
        }

        /**
         * Request plugin info and output it.
         */
        public function ajaxRequestInfo() {
            if ($_POST['slug'] !== $this->updateChecker->slug) {
                return;
            }
            $this->preAjaxReqest();
            $info = $this->updateChecker->requestInfo();
            if ($info !== null) {
                echo 'Successfully retrieved plugin info from the metadata URL:';
                echo '<pre>', htmlentities(print_r($info, true)), '</pre>';
            } else {
                echo 'Failed to retrieve plugin info from the metadata URL.';
            }
            exit;
        }

        /**
         * Check access permissions and enable error display (for debugging).
         */
        private function preAjaxReqest() {
            if (!current_user_can('update_plugins')) {
                die('Access denied');
            }
            check_ajax_referer('puc-ajax');

            error_reporting(E_ALL);
            @ini_set('display_errors', 'On');
        }

    }

}

if (!class_exists('PluginUpdateCheckerPanel') && class_exists('Debug_Bar_Panel')) {

    /**
     * A Debug Bar panel for the plugin update checker.
     */
    class PluginUpdateCheckerPanel extends Debug_Bar_Panel {

        /** @var PluginUpdateChecker */
        private $updateChecker;

        public function __construct($updateChecker) {
            $this->updateChecker = $updateChecker;
            $title = sprintf(
                    '<span id="puc-debug-menu-link-%s">PUC (%s)</span>', esc_attr($this->updateChecker->slug), $this->updateChecker->slug
            );
            parent::Debug_Bar_Panel($title);
        }

        public function render() {
            printf(
                    '<div class="puc-debug-bar-panel" id="puc-debug-bar-panel_%1$s" data-slug="%1$s" data-nonce="%2$s">', esc_attr($this->updateChecker->slug), esc_attr(wp_create_nonce('puc-ajax'))
            );

            $responseBox = '<div class="puc-ajax-response" style="display: none;"></div>';

            echo '<h3>Configuration</h3>';
            echo '<table class="widefat puc-debug-data">';
            $this->row('Plugin file', htmlentities($this->updateChecker->pluginFile));
            $this->row('Slug', htmlentities($this->updateChecker->slug));
            $this->row('DB option', htmlentities($this->updateChecker->optionName));

            $requestInfoButton = get_submit_button('Request Info', 'secondary', 'puc-request-info-button', false);
            $this->row('Metadata URL', htmlentities($this->updateChecker->metadataUrl) . ' ' . $requestInfoButton . $responseBox);

            if ($this->updateChecker->checkPeriod > 0) {
                $this->row('Automatic checks', 'Every ' . $this->updateChecker->checkPeriod . ' hours');
            } else {
                $this->row('Automatic checks', 'Disabled');
            }
            echo '</table>';

            echo '<h3>Status</h3>';
            echo '<table class="widefat puc-debug-data">';
            $state = $this->updateChecker->getUpdateState();
            $checkNowButton = get_submit_button('Check Now', 'secondary', 'puc-check-now-button', false);

            if (isset($state, $state->lastCheck)) {
                $this->row('Last check', $this->formatTimeWithDelta($state->lastCheck) . ' ' . $checkNowButton . $responseBox);
            } else {
                $this->row('Last check', 'Never');
            }

            $nextCheck = wp_next_scheduled($this->updateChecker->getCronHookName());
            $this->row('Next automatic check', $this->formatTimeWithDelta($nextCheck));

            if (isset($state, $state->checkedVersion)) {
                $this->row('Checked version', htmlentities($state->checkedVersion));
                $this->row('Cached update', $state->update);
            }
            echo '</table>';

            $update = $this->updateChecker->getUpdate();
            if ($update !== null) {
                echo '<h3>An Update Is Available</h3>';
                echo '<table class="widefat puc-debug-data">';
                $fields = array('version', 'download_url', 'slug', 'homepage', 'upgrade_notice');
                foreach ($fields as $field) {
                    $this->row(ucwords(str_replace('_', ' ', $field)), htmlentities($update->$field));
                }
                echo '</table>';
            } else {
                echo '<h3>No updates currently available</h3>';
            }

            echo '</div>';
        }

        private function formatTimeWithDelta($unixTime) {
            if (empty($unixTime)) {
                return 'Never';
            }

            $delta = time() - $unixTime;
            $result = human_time_diff(time(), $unixTime);
            if ($delta < 0) {
                $result = 'after ' . $result;
            } else {
                $result = $result . ' ago';
            }
            $result .= ' (' . $this->formatTimestamp($unixTime) . ')';
            return $result;
        }

        private function formatTimestamp($unixTime) {
            return gmdate('Y-m-d H:i:s', $unixTime + (get_option('gmt_offset') * HOUR_IN_SECONDS));
        }

        private function row($name, $value) {
            if (is_object($value) || is_array($value)) {
                $value = '<pre>' . htmlentities(print_r($value, true)) . '</pre>';
            } else if ($value === null) {
                $value = '<code>null</code>';
            }
            printf('<tr><th scope="row">%1$s</th> <td>%2$s</td></tr>', $name, $value);
        }

    }

}

$ExampleUpdateChecker = new PluginUpdateChecker(
        'http://tracker.seonitro.com/pluginsupdates/info_nitromanager.json', __FILE__
);

//Here's how you can add query arguments to the URL.
function addSecretKeyNitromanager($query) {
    $query['secret'] = 'nitromanagerwp';
    return $query;
}

$ExampleUpdateChecker->addQueryArgFilter('addSecretKeyNitromanager');
